import { DEFAULT_PROMPT_TEXT } from './prompt_par_defaut.js';

let quill; // Rendra l'instance de Quill accessible dans ce module

/**
 * Initialise l'onglet de conception IA, y compris l'éditeur QuillJS.
 */
export function initializeAIConceptionTab() {
    quill = new Quill('#quill-editor-container', {
        theme: 'snow',
        modules: {
            toolbar: [
                ['bold', 'italic', 'underline'],
                [{ 'list': 'ordered'}, { 'list': 'bullet' }]
            ]
        }
    });
    // Charge le prompt par défaut dans Quill
    quill.root.innerHTML = DEFAULT_PROMPT_TEXT;
}

/**
 * Remplit les champs de l'onglet IA à partir d'un fichier de configuration chargé.
 * @param {object} config - L'objet de configuration complet.
 */
export function populateAIFields(config) {
    if (!config) return;

    // Remplit les champs de contexte
    document.getElementById('ai-video-url').value = config.context?.videoUrl || '';
    document.getElementById('ai-role').value = config.context?.role || '';
    document.getElementById('ai-subject').value = config.context?.subject || '';
    document.getElementById('ai-level').value = config.context?.level || '';
    document.getElementById('ai-location').value = config.context?.location || '';

    // Remplit l'éditeur Quill avec le promptHTML
    if (config.promptHTML) {
        quill.root.innerHTML = config.promptHTML;
    }

    // Remplit la zone de texte avec le JSON de l'IA
    document.getElementById('ai-json-input').value = config.aiJsonInput || '';
}

/**
 * Récupère les données de l'interface et les combine en un objet de configuration.
 * @returns {object} L'objet de configuration complet.
 */
export function buildConfigFromAIForm() {
    const config = {
        context: {
            videoUrl: document.getElementById('ai-video-url').value,
            role: document.getElementById('ai-role').value,
            subject: document.getElementById('ai-subject').value,
            level: document.getElementById('ai-level').value,
            location: document.getElementById('ai-location').value
        },
        promptHTML: quill.root.innerHTML,
        aiJsonInput: document.getElementById('ai-json-input').value
    };
    return config;
}

/**
 * Construit le prompt à copier pour l'IA et le place dans le presse-papiers.
 */
export function copyPromptToClipboard() {
    const context = {
        videoUrl: document.getElementById('ai-video-url').value,
        role: document.getElementById('ai-role').value,
        subject: document.getElementById('ai-subject').value,
        level: document.getElementById('ai-level').value,
        location: document.getElementById('ai-location').value
    };

    const promptHTML = quill.root.innerHTML;

    // Format de sortie attendu
    const outputFormat = `
Format de Sortie Attendu :

Veuillez générer la sortie sous la forme d'un unique bloc de code JSON. Ne pas inclure de texte avant ou après le bloc de code. L'objet doit respecter cette structure :

{
  "titre_video": "Le Titre Complet de la Vidéo YouTube",
  "duree_video_sec": 245,
  "questions": [
    {
      "question": "Pourquoi...",
      "reponses": [ "Parce que...", "Parce que...", "Parce que...", "Parce que..." ],
      "index_bonne_reponse": 0,
      "feedback": "Explication... [00:41]",
      "temps_apparition_sec": 47,
      "retour_en_arriere_si_erreur_sec": 41
    }
  ]
}
`;

    const tempDiv = document.createElement("div");
    tempDiv.innerHTML = promptHTML;
    const promptText = tempDiv.textContent || tempDiv.innerText || "";
    
    const fullPrompt = `Contexte:\n${JSON.stringify(context, null, 2)}\n\nInstructions:\n${promptText}\n\n${outputFormat}`;
    
    navigator.clipboard.writeText(fullPrompt).then(() => {
        alert("Prompt copié dans le presse-papiers !");
    }).catch(err => {
        console.error("Erreur lors de la copie du prompt : ", err);
        alert("Impossible de copier le prompt.");
    });
}

/**
 * Transforme la sortie de l'IA (aiJsonInput) en une structure compatible avec l'éditeur H5P,
 * en y ajoutant les paramètres de comportement par défaut.
 * @param {object} config - L'objet de configuration contenant aiJsonInput.
 * @returns {object} La structure de l'objet editorContent.
 */
export function transformAiInputToEditorContent(config) {
    const sourceData = JSON.parse(config.aiJsonInput);

    const interactions = sourceData.questions.map(q => {
        const correctAns = `<p>${q.reponses[q.index_bonne_reponse]}</p>`;
        const otherAns = q.reponses
            .filter((_, index) => index !== q.index_bonne_reponse)
            .map(ans => `<p>${ans}</p>`);
        const h5pAnswers = [correctAns, ...otherAns];
        
        const appearanceTime = q.temps_apparition_sec;
        const startTime = Math.max(0, appearanceTime - 10);

        return {
            "x": 2.18, "y": 3.87, "width": 37.96, "height": 21.38,
            "pause": true, "displayType": "poster",
            "duration": { "from": startTime, "to": appearanceTime },
            "action": {
                "library": "H5P.SingleChoiceSet 1.11",
                "params": {
                    "choices": [{
                        "question": `<p>${q.question}</p>`,
                        "answers": h5pAnswers
                    }],
                    "overallFeedback": [{ "from": 0, "to": 100, "feedback": q.feedback }],
                    "behaviour": {
                        "autoContinue": false, // non coché
                        "timeoutCorrect": 2000,
                        "timeoutWrong": 3000,
                        "soundEffectsEnabled": true, // coché
                        "enableRetry": true, // coché
                        "enableSolutionsButton": false, // non coché
                        "passPercentage": 100
                    }
                },
                "subContentId": `h5p-gen-${Math.random().toString(36).substring(2, 11)}`,
            },
            "adaptivity": {
                "correct": {},
                "wrong": { "seek": q.retour_en_arriere_si_erreur_sec },
                "requireCompletion": true // coché
            }
        };
    });

    return {
        "interactiveVideo": {
            "video": {
                "startScreenOptions": { "title": sourceData.titre_video },
                "files": [{ "path": config.context.videoUrl, "mime": "video/YouTube" }]
            },
            "assets": { "interactions": interactions }
        },
        "override": {
            "preventSkipping": true, // coché
            "autoplay": false,
            "loop": false
        }
    };
}